#!/bin/bash

# Some utility functions to be used in the install scripts

# Prompt user until they type Y/y or N/n
# Calling:
# prompt_yesno
prompt_yesno() {
	local input=""

	while true; do
		echo -n "(Y/n): "
		read input

		if [[ "$input" == "" || "$input" == "Y" || "$input" == "y" ]]; then
			return 0
		fi

		if [[ "$input" == "N" || "$input" == "n" ]]; then
			return 1
		fi
	done
}

# Check if version A is greater than or equal to version B
# Returns 0 if true, 1 if false
# Calling:
# is_version_a_gte_version_b "X.Y.Z" "A.B.C"
# By virtue of ignoring anything after element 2, it can handle version strings with garbage on the end
# It will allow periods, spaces, or dashes as number separators
is_version_a_gte_version_b() {
	# Return 1 when one of the two parameters isn't a version
	if ! [[ "$1" =~ [0-9]*[.\ -][0-9]*[.\ -][0-9]*.* ]]; then
		return 1
	fi

	if ! [[ "$2" =~ [0-9]*[.\ -][0-9]*[.\ -][0-9]*.* ]]; then
		return 1
	fi

	local versionA=($(echo "$1" | tr "." " " | tr "-" " "))
	local versionB=($(echo "$2" | tr "." " " | tr "-" " "))

	if [[ ${versionA[0]} -lt ${versionB[0]} ]]; then
		return 1
	elif [[ ${versionA[0]} -gt ${versionB[0]} ]]; then
		return 0
	else
		if [[ ${versionA[1]} -lt ${versionB[1]} ]]; then
			return 1
		elif [[ ${versionA[1]} -gt ${versionB[1]} ]]; then
			return 0
		else
			if [[ ${versionA[2]} -lt ${versionB[2]} ]]; then
				return 1
			elif [[ ${versionA[2]} -gt ${versionB[2]} ]]; then
				return 0
			else
				return 0
			fi
		fi
	fi
}

function make_backup() {
	# Make a temporary backup folder
	backup_temp_directory=$(mktemp -d /tmp/screenconnect-XXX)

	# Some files and folders are intentionally left in place during an upgrade - these are listed below
	# Grep for Directories to exclude
	grepExcludeDirectories="grep -v \"App_Data/Session\" | grep -v \"App_Data/Toolbox\""
	# Grep for Files to exclude
	grepExcludeFiles="grep -v \"App_Data/License.xml\" | grep -v \"App_Data/Role.xml\" | grep -v \"App_Data/User.xml\" | grep -v \"App_Data/Session.db\""

	# Move into the installation directory, after saving the current directory
	local last_working_directory=`pwd`
	cd "$installation_directory"

	# Create all folders that are in installation folder in backup folder
	eval "find ./ -type d 2>/dev/null | $grepExcludeDirectories | xargs -I{} mkdir $backup_temp_directory/{} &>/dev/null"
	
	# Copy files we need to back up always
	mv -f "$installation_directory/web.config" "$backup_temp_directory" &>/dev/null
	mv -f "$installation_directory/App_GlobalResources/Default.resx" "$backup_temp_directory/App_GlobalResources/" &>/dev/null
	mv -f "$installation_directory/App_ClientConfig/Elsinore.ScreenConnect.Client.resx" "$backup_temp_directory/App_ClientConfig/" &>/dev/null
	mv -f "$installation_directory/App_Data/SessionGroup.xml" "$backup_temp_directory/App_Data/SessionGroup.xml" &>/dev/null

	# Linux doesn't set creation dates
	# Get oldest file and figure out how long ago it was modified
	local oldest_file_mtime
	get_oldest_file_epoch_time "$installation_directory/Bin"
	
	# Set the target time to 1 minute after the oldest file was last modified
	# We will use this to differentiate installation files vs user-modified files
	target_mtime=$(expr $(date +%s) - $oldest_file_mtime)
	target_mtime=$(expr $target_mtime / 60 - 1)

	# Move all files modified since target_mtime ago of the oldest file to backup directory except excluded
	eval "find ./ -type f -mmin -$target_mtime 2>/dev/null | $grepExcludeFiles | $grepExcludeDirectories | xargs -I{} mv {} $backup_temp_directory/{} &>/dev/null"
	# Remove any remaining files except excluded
	eval "find ./ -type f 2>/dev/null | $grepExcludeFiles | $grepExcludeDirectories | xargs -I{} rm -f {}"
	# Remove any remaining empty directories, except excluded directories
	# (this will retain folders the excluded files are in, also)
	eval "find ./ -type d 2>/dev/null | $grepExcludeDirectories | xargs -I{} rmdir {} &>/dev/null"
	
	# Go back to the old working directory
	cd "$last_working_directory"
}

function restore_backup() {
	# Pull files that are always backed up
	mv -f "$backup_temp_directory/web.config" "$installation_directory" &>/dev/null
	mv -f "$backup_temp_directory/App_GlobalResources/Default.resx" "$installation_directory/App_GlobalResources/" &>/dev/null
	mv -f "$backup_temp_directory/App_ClientConfig/Elsinore.ScreenConnect.Client.resx" "$installation_directory/App_ClientConfig/" &>/dev/null
	mv -f "$backup_temp_directory/App_Data/SessionGroup.xml" "$installation_directory/App_Data/SessionGroup.xml" &>/dev/null

	# Bring in logo if it's there - a lot of customers modify it
	mv -f "$backup_temp_directory/Images/Logo.png" "$installation_directory/Images/Logo.png" &>/dev/null
	if [ $? -eq 0 ]; then
		# If we bring back the logo, make its modified time 5 minutes future so it gets backed up next upgrade
		update_mtime_to_5min_future "$installation_directory/Images/Logo.png" &>/dev/null
	fi
	
	# Remove all empty directories
	find "$backup_temp_directory" -depth -type d | xargs -I{} rmdir {} &>/dev/null

	if [ -e "$backup_temp_directory" ]; then
		echo "The upgrade process has detected that some files had been modified since last installation."
		echo "These files were saved for review in: $backup_temp_directory"
	fi
}

get_mono_executable_file_path() {
	find "$1" -name mono_* | sort > /tmp/monos

	while read monoFile; do
		"$monoFile" >/dev/null 2>&1

		if [ $? -eq 1 ]; then
			echo $monoFile
			break
		fi
	done < /tmp/monos

	rm /tmp/monos
}

# The next several functions are common utility functions for scripts in the Distributions folder

base_linux_dependency_check() {
	ldconfig -p | grep "libsqlite3\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libsqlite3 (critical): For accessing the session database"
	fi
	
	ldconfig -p | grep "libz\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libz (critical): For compressing web requests, building installers, and encoding video"
	fi
	
	ldconfig -p | grep "libavcodec\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libavcodec53 (optional): For transcoding videos when using Extended Auditing"
	else
		# If it's available, test if x264 support was built in
		grep "x264_encoder_encode" `ldconfig -p | grep "libavcodec\." -m 1 | awk '{ print $NF }'` &>/dev/null
		if [ $? -ne 0 ]; then
			problematic_dependencies[${#problematic_dependencies[*]}]="libavcodec53 (optional): For transcoding videos when using Extended Auditing"
		fi
	fi

	ldconfig -p | grep "libswscale\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libswscale2 (optional): For transcoding videos when using Extended Auditing"
	fi

	ldconfig -p | grep "libavutil\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libavutil51 (optional): For transcoding videos when using Extended Auditing"
	fi

	ldconfig -p | grep "libavformat\." &>/dev/null
	if [ $? -ne 0 ]; then
		problematic_dependencies[${#problematic_dependencies[*]}]="libavformat53 (optional): For transcoding videos when using Extended Auditing"
	fi
}

# Gets the oldest file's mtime in the provided folder (does not recurse)
# Calling: 
# base_linux_get_oldest_file_epoch_time $directory
base_linux_get_oldest_file_epoch_time() {
	local directory=$1
	oldest_file_mtime=$(stat -c %Y "$directory/$(ls -t "$directory" | tail -1)")
}

# Set a file's modified time to 5 minutes in the future
# Calling:
# base_linux_update_mtime_to_5min_future $file
base_linux_update_mtime_to_5min_future() {
	local folder=$1
	touch -d "@$(date --date '5min')" "$installation_directory/Images/Logo.png" &>/dev/null
}

# Build an action to start ScreenConnect
base_linux_build_start_service_action() {
	installer_actions[${#installer_actions[*]}]="\"$initd_directory/$service_name\" start >/dev/null"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Start $service_name service"
}

# Build an action to stop ScreenConnect, remove the init.d script, and remove any rc.d links
base_linux_build_stop_and_remove_service_action() {
	local servicename=$1
	local filepath="$initd_directory/$1"

	# Stop the service
	installer_actions[${#installer_actions[*]}]="\"$filepath\" stop &>/dev/null"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Stop existing service $servicename if running"

	# Removing rc.d links
	installer_actions[${#installer_actions[*]}]="find -L $rcd_directory/rc?.d -samefile \"$filepath\" 2>/dev/null | xargs -I{} rm {}"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Remove startup links pointing to script at $filepath"
	
	# Remove script itself
	installer_actions[${#installer_actions[*]}]="rm \"$filepath\" -f"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Remove service start script at $filepath"
}

# Grep an init.d directory for an identifying line pointing to the installation directory
# Assumes $initd_directory and $installation_directory are defined
# Fills service_names array
base_linux_get_existing_service_names() {
	service_names=($(grep -l "$installation_directory" "$initd_directory/"* 2>/dev/null | xargs -I{} basename {} | tr "\n" " "))
}

# Install to init.d/rc.d in linux
# Assumes variables are global
base_linux_build_startup_actions() {
	if [ "$service_name" == "" ]; then
		echo "Service name is blank, setting to \"screenconnect\"!"
		service_name="screenconnect"
	fi

	# Setup init.d
	# Use ,'s as the sed separator since installation_directory will contain slashes
	INITD_SCRIPT="screenconnect"
	installer_actions[${#installer_actions[*]}]="
		cp \"$INSTALL_DIRECTORY/$INITD_SCRIPT\" \"$INSTALL_DIRECTORY/$INITD_SCRIPT.old\";
		sed -e \"s,SCREENCONNECT_PATH=\\\".*\\\",SCREENCONNECT_PATH=\\\"$installation_directory\\\",\" < \"$INSTALL_DIRECTORY/$INITD_SCRIPT.old\" > \"$INSTALL_DIRECTORY/$INITD_SCRIPT\";
		cp \"$INSTALL_DIRECTORY/$INITD_SCRIPT\" \"$INSTALL_DIRECTORY/$INITD_SCRIPT.old\";
		rm \"$INSTALL_DIRECTORY/$INITD_SCRIPT.old\";
		cp \"$INSTALL_DIRECTORY/$INITD_SCRIPT\" \"$initd_directory/$service_name\";
		chmod 755 \"$initd_directory/$service_name\";
	"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Create service script at $initd_directory/$service_name"

	# Setup rc.d
	# Runlevels 2,3,5 are startup with various options (GUI, network)
	# Runlevel 0 is halt, 6 is reboot
	local rcd_link_name="03$service_name"
	installer_actions[${#installer_actions[*]}]="
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc2.d/S$rcd_link_name\" &>/dev/null;
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc3.d/S$rcd_link_name\" &>/dev/null;
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc4.d/S$rcd_link_name\" &>/dev/null;
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc5.d/S$rcd_link_name\" &>/dev/null;
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc0.d/K$rcd_link_name\" &>/dev/null;
		ln -s \"$initd_directory/$service_name\" \"$rcd_directory/rc6.d/K$rcd_link_name\" &>/dev/null;
	"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Create startup links in $rcd_directory/rcX.d/ directories"
}

base_linux_launch_web_page() {
	echo "To access your new ScreenConnect installation, open a browser and navigate to:"
	echo "$1"
}
