﻿using System;
using System.Collections;
using System.Configuration;
using System.Linq;
using System.Web.Configuration;
using System.Web.Security;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Collections.Generic;

namespace Elsinore.ScreenConnect
{
	public partial class SecurityPanel : UserControl, IDataControl
	{
		protected const string EmptyPassword = "CE5B9879";

		AuthenticationMode authenticationMode;

		protected override void LoadViewState(object savedState)
		{
			var mainState = (Triplet)savedState;
			base.LoadViewState(mainState.First);
			this.authenticationMode = (AuthenticationMode)mainState.Second;

			var gridState = (Pair)mainState.Third;
			this.userView.DataSource = gridState.First;
			this.roleView.DataSource = gridState.Second;
			this.userView.DataBind();
			this.roleView.DataBind();
		}

		protected override object SaveViewState()
		{
			var state = new Triplet();
			state.First = base.SaveViewState();
			state.Second = this.authenticationMode;
			state.Third = new Pair(this.userView.DataSource, this.roleView.DataSource);
			return state;
		}

		protected override void OnPreRender(EventArgs e)
		{
			base.OnPreRender(e);

			var otherAuthenticationMode = (this.authenticationMode == AuthenticationMode.Forms ? AuthenticationMode.Windows : AuthenticationMode.Forms);

			this.authenticationTitleLabel.InnerHtml = this.currentAuthenticationTitlePanel.InnerHtml = Resources.Default.ResourceManager.GetString(string.Format("SecurityPanel.{0}AuthenticationTitle", this.authenticationMode));
			this.currentAuthenticationDescriptionPanel.InnerHtml = Resources.Default.ResourceManager.GetString(string.Format("SecurityPanel.{0}AuthenticationDescription", this.authenticationMode));
			this.otherAuthenticationTitlePanel.InnerHtml = Resources.Default.ResourceManager.GetString(string.Format("SecurityPanel.{0}AuthenticationTitle", otherAuthenticationMode));
			this.otherAuthenticationDescriptionPanel.InnerHtml = Resources.Default.ResourceManager.GetString(string.Format("SecurityPanel.{0}AuthenticationDescription", otherAuthenticationMode));

			this.usersLabel.Visible = this.usersPanel.Visible = (this.authenticationMode == AuthenticationMode.Forms && this.userView.DataSource != null);

			this.windowsInstructionsPanel.Visible = (this.authenticationMode == AuthenticationMode.Windows);

			this.newRoleButton.Enabled =
			this.newUserButton.Enabled = (this.roleView.EditIndex == -1 && this.userView.EditIndex == -1);

			this.changeButton.Visible = ServerToolkit.Instance.IsWindowsAuthenticationSupported();

			this.userView.Columns.OfType<BoundField>().First(bf => bf.DataField == "PasswordQuestion").Visible = Elsinore.ScreenConnect.Extensions.TryParseBool(Resources.Default.SecurityPanel_PasswordQuestionVisible);
			this.userView.Columns.OfType<BoundField>().First(bf => bf.DataField == "Comment").Visible = Elsinore.ScreenConnect.Extensions.TryParseBool(Resources.Default.SecurityPanel_CommentVisible);
		}

		protected void OnAuthenticationButtonClick(object sender, EventArgs e)
		{
			var commandName = ((IButtonControl)sender).CommandName;

			if (commandName == "Change")
			{
				this.multiView.ActiveViewIndex = 1;
			}
			else if (commandName == "Stay")
			{
				this.multiView.ActiveViewIndex = 0;
			}
			else if (commandName == "Switch")
			{
				this.authenticationMode = (this.authenticationMode == AuthenticationMode.Forms ? AuthenticationMode.Windows : AuthenticationMode.Forms);

				if (((IButtonControl)sender).CommandArgument == "Default")
				{
					this.roleView.DataSource = PermissionInfo.GetDefaultRoles(this.authenticationMode).Select(r => new GridRole { Name = r.Name, OriginalName = r.Name, PermissionEntries = new List<PermissionEntry>(r.PermissionEntries) }).ToArray();
					this.roleView.DataBind();
				}

				if (this.authenticationMode == AuthenticationMode.Forms)
				{
					this.BindFormsUsers(this.roleView.DataSource.Cast<GridRole>().Select(gr => gr.Name));
				}

				this.multiView.ActiveViewIndex = 0;
			}
		}

		protected void OnViewRowEditing(object sender, GridViewEditEventArgs e)
		{
			this.BeginEdit((GridView)sender, e.NewEditIndex);
		}

		protected void OnViewRowCancelingEdit(object sender, GridViewCancelEditEventArgs e)
		{
			this.CancelEditAll();
		}

		protected void OnViewRowUpdating(object sender, GridViewUpdateEventArgs e)
		{
			this.EndEdit((GridView)sender);
		}

		protected void OnViewRowDeleting(object sender, GridViewDeleteEventArgs e)
		{
			this.CancelEditAll();

			var gridView = (GridView)sender;

			if (gridView == this.roleView)
			{
				foreach (var gridUser in this.userView.DataSource.Cast<GridUser>())
					gridUser.Roles = gridUser.Roles.Where(r => r != ((GridNamedItem[])gridView.DataSource)[e.RowIndex].Name).ToList();
			}

			gridView.DataSource = ((GridNamedItem[])gridView.DataSource).Where((gu, index) => index != e.RowIndex).ToArray();
			this.FindControls<GridView>().ForEach(gv => gv.DataBind());

			WebExtensions.RegisterUpdatePanelStartupScript(this, "SC.ui.setChangesMade();");
		}

		protected void OnNameValidatorServerValidate(object sender, ServerValidateEventArgs e)
		{
			var gridView = WebExtensions.FindAncestor<GridView>((Control)sender);
			var name = (e.Value == string.Empty ? null : e.Value);
			e.IsValid = !((GridNamedItem[])gridView.DataSource).Where((gi, index) => index != gridView.EditIndex && (string.Equals(gi.Name, name, StringComparison.InvariantCultureIgnoreCase) || string.Equals(gi.OriginalName, name, StringComparison.InvariantCultureIgnoreCase))).Any();
		}

		protected void OnNewUserButtonClick(object sender, EventArgs e)
		{
			var newUser = new GridUser { Roles = new List<string>() };
			this.EditNewItem(this.userView, newUser);
		}

		protected void OnNewRoleButtonClick(object sender, EventArgs e)
		{
			var newRole = new GridRole { PermissionEntries = new List<PermissionEntry>() };
			this.EditNewItem(this.roleView, newRole);
		}

		protected void OnUserValidatorServerValidate(object sender, ServerValidateEventArgs e)
		{
			var validator = (CustomValidator)sender;

			if (!this.ValidateGrid(this.userView))
			{
				e.IsValid = false;
				validator.Text = Resources.Default.SecurityPanel_MustCompleteErrorMessage;
			}

			var administerRoleNames = ((GridNamedItem[])this.roleView.DataSource).Cast<GridRole>().Where(gr => gr.PermissionEntries.Any(pe => pe.Name == PermissionInfo.AdministerPermission)).Select(r => r.Name);

			if (e.IsValid && !this.userView.DataSource.Cast<GridUser>().Any(u => u.Roles.Intersect(administerRoleNames).Any()))
			{
				e.IsValid = false;
				validator.Text = Resources.Default.SecurityPanel_MustDefineAdministerUser;
			}
		}

		protected void OnRoleValidatorServerValidate(object sender, ServerValidateEventArgs e)
		{
			var validator = (CustomValidator)sender;

			if (!this.ValidateGrid(this.roleView))
			{
				e.IsValid = false;
				validator.Text = Resources.Default.SecurityPanel_MustCompleteErrorMessage;
			}

			if (e.IsValid && !this.roleView.DataSource.Cast<GridRole>().Any(gr => gr.PermissionEntries.Any(pe => pe.Name == PermissionInfo.AdministerPermission)))
			{
				e.IsValid = false;
				validator.Text = Resources.Default.SecurityPanel_MustDefineAdministerRole;
			}
		}

		protected void OnRepeaterItemCommand(object sender, RepeaterCommandEventArgs e)
		{
			var repeater = (RepeaterEx)sender;

			if (e.CommandName == "Delete")
			{
				((IList)repeater.DataSource).RemoveAt(e.Item.ItemIndex);
				repeater.Rebind();

				if (repeater.FindAncestor<GridView>() == this.userView)
				{
					var gridViewRow = repeater.FindAncestor<GridViewRow>();
					gridViewRow.FindControl("rolePanel").DataBind();
				}
			}
		}

		protected void OnPermissionListChanged(object sender, EventArgs e)
		{
			this.SetPermissionControlVisibility((Control)sender);
		}

		protected void OnPermissionPanelDataBinding(object sender, EventArgs e)
		{
			var panel = (Control)sender;

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
			{
				this.PopulateStringListItems(panel, "permissionList", PermissionInfo.GetPermissionNames());
				this.PopulateStringListItems(panel, "sessionGroupFilterList", Enum.GetNames(typeof(SessionGroupFilter)));
				this.PopulateStringListItems(panel, "sessionGroupList", sessionManager.GetSessionGroups().Select(sg => sg.Name));
				this.PopulateStringListItems(panel, "ownershipFilterList", Enum.GetNames(typeof(OwnershipFilter)));
			}

			this.SetPermissionControlVisibility(panel);
		}

		protected void OnRolePanelDataBinding(object sender, EventArgs e)
		{
			var panel = (Control)sender;
			var editingRoles = (IEnumerable<string>)panel.FindAncestor<GridViewRow>().FindControl<RepeaterEx>().DataSource;
			panel.Visible = this.PopulateStringListItems(panel, "roleList", ((GridNamedItem[])this.roleView.DataSource).Where(r => !editingRoles.Contains(r.Name)).Select(r => r.Name)) != 0;
		}

		protected void OnAddButtonClick(object sender, EventArgs e)
		{
			var addButton = (Control)sender;
			var gridViewRow = addButton.FindAncestor<GridViewRow>();
			var gridView = gridViewRow.FindAncestor<GridView>();
			var item = ((GridNamedItem[])gridView.DataSource)[gridView.EditIndex];

			if (gridView == this.userView)
			{
				var roleName = ((DropDownList)addButton.FindControl("roleList")).SelectedValue;
				((GridUser)item).EditingRoles.Add(roleName);
				gridViewRow.FindControl("rolePanel").DataBind();
			}

			if (gridView == this.roleView)
			{
				var permissionName = ((DropDownList)addButton.FindControl("permissionList")).SelectedValue;
				var sessionGroupFilter = Elsinore.ScreenConnect.Extensions.TryParseEnum<SessionGroupFilter>(((DropDownList)addButton.FindControl("sessionGroupFilterList")).SelectedValue);
				var ownershipFilter = Elsinore.ScreenConnect.Extensions.TryParseEnum<OwnershipFilter>(((DropDownList)addButton.FindControl("ownershipFilterList")).SelectedValue);
				var sessionGroupName = (sessionGroupFilter != SessionGroupFilter.SpecificSessionGroup ? null : ((DropDownList)addButton.FindControl("sessionGroupList")).SelectedValue);
				var permissionEntry = PermissionInfo.CreatePermissionEntry(permissionName, sessionGroupFilter, ownershipFilter, sessionGroupName);
				((GridRole)item).EditingPermissionEntries.Add(permissionEntry);
			}

			gridViewRow.FindControl<RepeaterEx>().Rebind();
		}

		protected void OnSecurityTypeCheckChanged(object sender, EventArgs e)
		{
			this.CancelEditAll();
		}

		bool ValidateGrid(GridView gridView)
		{
			if (gridView.EditIndex == -1)
				return true;

			this.Page.Validate("Grid");

			if (!this.Page.IsValid)
				return false;

			this.EndEdit(gridView);
			return true;

		}

		int PopulateStringListItems(Control baseControl, string controlName, IEnumerable<string> items)
		{
			var control = (ListControl)baseControl.FindControl(controlName);
			control.Items.Clear();
			items.ForEach(item => control.Items.Add(item));
			return control.Items.Count;
		}

		void SetPermissionControlVisibility(Control control)
		{
			bool sessionGroupFilterRelevant, ownershipFilterRelevant, sessionGroupNameRelevant;
			var permissionList = (DropDownList)control.FindControl("permissionList");
			var sessionGroupFilterList = (DropDownList)control.FindControl("sessionGroupFilterList");
			var sessionGroupFilter = Elsinore.ScreenConnect.Extensions.TryParseEnum<SessionGroupFilter>(sessionGroupFilterList.SelectedValue);
			PermissionInfo.CalculateFieldRelevance(permissionList.SelectedValue, sessionGroupFilter, string.Empty, out sessionGroupFilterRelevant, out ownershipFilterRelevant, out sessionGroupNameRelevant);
			sessionGroupFilterList.Visible = sessionGroupFilterRelevant;
			((Control)control.FindControl("sessionGroupList")).Visible = sessionGroupNameRelevant;
			((Control)control.FindControl("ownershipFilterList")).Visible = ownershipFilterRelevant;
		}

		void EditNewItem(GridView gridView, GridNamedItem item)
		{
			var items = (GridNamedItem[])gridView.DataSource;
			gridView.DataSource = items.Concat(new GridNamedItem[] { item }).ToArray();
			this.BeginEdit(gridView, items.Length);
		}

		void BeginEdit(GridView gridView, int index)
		{
			this.CancelEditAll();

			if (gridView == this.userView)
			{
				var gridUser = (GridUser)((GridNamedItem[])gridView.DataSource)[index];
				gridUser.EditingRoles = gridUser.Roles.ToList();
			}

			if (gridView == this.roleView)
			{
				var gridRole = (GridRole)((GridNamedItem[])gridView.DataSource)[index];
				gridRole.EditingPermissionEntries = gridRole.PermissionEntries.ToList();
			}

			gridView.EditIndex = index;
			this.FindControls<GridView>().ForEach(gv => gv.DataBind());
		}

		void EndEdit(GridView gridView)
		{
			var items = (GridNamedItem[])gridView.DataSource;
			var item = items[gridView.EditIndex];
			var row = gridView.Rows[gridView.EditIndex];

			var oldName = item.Name;
			item.Name = row.Cells[1].Controls.OfType<TextBox>().First().GetTrimmedOrNullText();

			if (gridView == this.userView)
			{
				((GridUser)item).Password = this.GetCellEditText(row, 2);
				((GridUser)item).PasswordQuestion = this.GetCellEditText(row, 3);
				((GridUser)item).Comment = this.GetCellEditText(row, 4);
				((GridUser)item).Roles = ((GridUser)item).EditingRoles;
			}

			if (gridView == this.roleView)
			{
				((GridRole)item).PermissionEntries = ((GridRole)item).EditingPermissionEntries;

				if (oldName != item.Name)
				{
					foreach (var gridUser in this.userView.DataSource.Cast<GridUser>())
					{
						var roleIndex = gridUser.Roles.IndexOf(oldName);

						if (roleIndex != -1)
						{
							gridUser.Roles.RemoveAt(roleIndex);
							gridUser.Roles.Insert(roleIndex, item.Name);
						}
					}
				}
			}

			WebExtensions.RegisterUpdatePanelStartupScript(this, "SC.ui.setChangesMade();");

			gridView.EditIndex = -1;
			this.FindControls<GridView>().ForEach(gv => gv.DataBind());
		}

		string GetCellEditText(GridViewRow row, int cellIndex)
		{
			return row.Cells[cellIndex].Controls.OfType<TextBox>().First().GetTrimmedOrNullText();
		}

		void CancelEditAll()
		{
			this.FindControls<GridView>().ForEach(gv => this.TryCancelEdit(gv));
			this.FindControls<GridView>().ForEach(gv => gv.DataBind());
		}

		void TryCancelEdit(GridView gridView)
		{
			if (gridView.EditIndex != -1)
			{
				var items = (GridNamedItem[])gridView.DataSource;

				if (items[gridView.EditIndex].OriginalName == null)
				{
					gridView.DataSource = items.Where((gu, index) => index != gridView.EditIndex).ToArray();
				}
				else
				{
					if (gridView == this.userView)
						((GridUser)((GridNamedItem[])gridView.DataSource)[gridView.EditIndex]).EditingRoles = null;

					if (gridView == this.roleView)
						((GridRole)((GridNamedItem[])gridView.DataSource)[gridView.EditIndex]).EditingPermissionEntries = null;
				}

				gridView.EditIndex = -1;
			}
		}

		void BindFormsUsers(IEnumerable<string> allowedRoles)
		{
			var roleProvider = Roles.Providers[AuthenticationMode.Forms.ToString()].AssertNonNull();

			try
			{
				this.userView.DataSource = WebExtensions.GetAllMembershipUsers().Select(m => new GridUser
				{
					OriginalName = m.UserName,
					Name = m.UserName,
					PasswordQuestion = m.PasswordQuestion,
					Comment = m.Comment,
					Password = SecurityPanel.EmptyPassword,
					Roles = new List<string>(roleProvider.GetRolesForUser(m.UserName).Intersect(allowedRoles))
				}).OfType<GridNamedItem>().ToArray();
			}
			catch (NotSupportedException)
			{
				this.userView.DataSource = null;
			}

			this.userView.DataBind();
		}

		public void Refresh()
		{
			var configuration = WebConfigurationManager.OpenWebConfiguration();
			this.authenticationMode = WebConfigurationManager.GetSection<AuthenticationSection>(configuration).Mode;

			var gridRoles = Permissions.Provider.GetAllRoles().Select(r => new GridRole
			{
				OriginalName = r.Name,
				Name = r.Name,
				PermissionEntries = new List<PermissionEntry>(r.PermissionEntries)
			}).OfType<GridNamedItem>().ToArray();

			this.roleView.DataSource = gridRoles;
			this.roleView.DataBind();

			if (this.authenticationMode == AuthenticationMode.Forms)
				this.BindFormsUsers(gridRoles.Select(gr => gr.Name));

			WebExtensions.RegisterUpdatePanelStartupScript(this, "SC.ui.clearChangesMade();");
		}

		public void Save()
		{
			var configuration = WebConfigurationManager.OpenWebConfiguration();
			var authenticationSection = WebConfigurationManager.GetSection<AuthenticationSection>(configuration);
			var roleManagerSection = WebConfigurationManager.GetSection<RoleManagerSection>(configuration);

			if (authenticationSection.Mode != this.authenticationMode || roleManagerSection.DefaultProvider != this.authenticationMode.ToString())
			{
				authenticationSection.Mode = this.authenticationMode;
				roleManagerSection.DefaultProvider = this.authenticationMode.ToString();
				ServerToolkit.Instance.SaveConfiguration(configuration);
			}

			var gridRoles = (GridNamedItem[])this.roleView.DataSource;
			var roles = gridRoles.Cast<GridRole>().Select(gr => new Role { Name = gr.Name, PermissionEntries = gr.PermissionEntries.ToArray() });
			Permissions.Provider.SaveRoles(roles);

			if (this.authenticationMode == AuthenticationMode.Forms && this.userView.DataSource != null)
			{
				var users = (GridNamedItem[])this.userView.DataSource;
				var roleProvider = Roles.Providers[roleManagerSection.DefaultProvider].AssertNonNull();

				foreach (var existingMembershipUser in WebExtensions.GetAllMembershipUsers())
					if (!users.Any(u => u.OriginalName == existingMembershipUser.UserName))
						Membership.DeleteUser(existingMembershipUser.UserName);

				foreach (var gridUser in users.Cast<GridUser>())
				{
					if (gridUser.OriginalName == null)
					{
						Membership.CreateUser(gridUser.Name, gridUser.Password);
					}
					else
					{
						if (gridUser.OriginalName != gridUser.Name)
							if (!((IChangeUserNameProvider)Membership.Provider).ChangeUserName(gridUser.OriginalName, gridUser.Name))
								throw new InvalidOperationException("Unable to change user name for: " + gridUser.OriginalName);

						if (gridUser.Password != SecurityPanel.EmptyPassword)
							if (!Membership.Provider.ChangePassword(gridUser.Name, null, gridUser.Password))
								throw new InvalidOperationException("Unable to change password for: " + gridUser.Name);
					}

					if (!Membership.Provider.ChangePasswordQuestionAndAnswer(gridUser.Name, null, gridUser.PasswordQuestion, null))
						throw new InvalidOperationException("Unable to change password question for: " + gridUser.Name);

					var user = Membership.Provider.GetUser(gridUser.Name, false);
					user.Comment = gridUser.Comment;
					Membership.Provider.UpdateUser(user);

					var userRoles = roleProvider.GetRolesForUser(gridUser.Name);

					if (userRoles.Length != gridUser.Roles.Count || userRoles.Intersect(gridUser.Roles).Count() != userRoles.Length)
					{
						roleProvider.RemoveUsersFromRoles(new[] { gridUser.Name }, userRoles);
						roleProvider.AddUsersToRoles(new[] { gridUser.Name }, gridUser.Roles.ToArray());
					}
				}
			}

			WebExtensions.RegisterUpdatePanelStartupScript(this, "SC.ui.clearChangesMade();");
		}

		public bool IsSaveAllowed { get { return this.multiView.ActiveViewIndex == 0; } }
	}
}