﻿<%@ WebHandler Language="C#" Class="Elsinore.ScreenConnect.Service" %>

using System;
using System.Web;
using System.Collections.Generic;
using System.Linq;
using System.Configuration;
using System.Net.Mail;
using System.Net.Configuration;

namespace Elsinore.ScreenConnect
{
	public class Service : WebServiceBase
	{
		public IAsyncResult BeginGetHostSessionInfo(string sessionGroupName, string additionalFilter, Guid? findSessionID, long version, AsyncCallback callback, object state)
		{
			var permissionEntries = Permissions.GetEntriesForUser();
			var userName = HttpContext.Current.User.Identity.Name;

			Permissions.AssertPermission(new PermissionRequest(), permissionEntries, true);

			return new WaitForChangeAsyncResult(callback, state, HttpContext.Current, version, delegate(long newVersion)
			{
				var utcNow = DateTime.UtcNow;
				var variables = this.GetVariables(userName);

				using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				{
					sessionManager.EnsureEligibleHost(userName);

					var sessionGroupSummaries = sessionManager.GetSessionGroupSummaries(variables);

					if (!Permissions.HasPermission(PermissionInfo.ViewSessionGroupPermission, permissionEntries))
						sessionGroupSummaries = sessionGroupSummaries.Where(sgs => Permissions.HasPermission(new SessionPermissionRequest { Name = PermissionInfo.ViewSessionGroupPermission, SessionType = sgs.SessionType, SessionGroupName = sgs.Name }, permissionEntries)).ToArray();

					var sessionGroupIndex = Array.FindIndex(sessionGroupSummaries, sgs => string.Equals(sgs.Name, sessionGroupName, StringComparison.InvariantCultureIgnoreCase));

					if (findSessionID != null)
					{
						var sessionGroupNames = sessionGroupSummaries.Select(sgs => sgs.Name).ToArray();
						var foundSessionGroupIndex = sessionManager.FindFirstSessionGroupWithSession(sessionGroupNames, variables, findSessionID.Value, sessionGroupIndex);
						if (foundSessionGroupIndex != -1) sessionGroupIndex = foundSessionGroupIndex;
					}

					if (sessionGroupIndex == -1 && sessionGroupSummaries.Length != 0)
						sessionGroupIndex = 0;

					var sessions = default(IEnumerable<Session>);

					if (sessionGroupIndex == -1 || (sessionGroupSummaries[sessionGroupIndex].LastSetAlteredVersion <= version && sessionGroupSummaries[sessionGroupIndex].LastSessionAlteredVersion <= version))
					{
						sessions = new Session[0];
					}
					else
					{
						sessions = sessionManager.GetSessions(sessionGroupSummaries[sessionGroupIndex].Name, variables, additionalFilter);

						if (sessionGroupSummaries[sessionGroupIndex].LastSetAlteredVersion <= version)
							sessions = sessions.Where(s => s.LastAlteredVersion > version);
					}

					var permissionSets = new LazyList<SessionPermissions>(2, index => PermissionInfo.GetSessionPermissions(permissionEntries, sessionGroupSummaries[sessionGroupIndex].SessionType, sessionGroupSummaries[sessionGroupIndex].Name, index == 0));

					return new
					{
						v = newVersion,
						sgi = sessionGroupIndex,
						sgs = sessionGroupSummaries.Select(sgs => new
						{
							n = sgs.Name,
							st = sgs.SessionType,
							sc = sgs.SessionCount,
							tv = sgs.LastSetAlteredVersion,
						}),
						sss = sessions.Select(s => new
						{
							st = s.SessionType,
							h = s.Host,
							ip = s.IsPublic,
							c = s.Code,
							cps = s.CustomPropertyValues,
							v = s.LastAlteredVersion,
							gun = s.GuestInfo.LoggedOnUserName,
							gud = s.GuestInfo.LoggedOnUserDomain,
							git = this.GetDurationSeconds(s.GuestInfo.LastActivityTime, utcNow),
							gos = s.GuestInfo.OperatingSystemName,
							gov = s.GuestInfo.OperatingSystemVersion.ToShortString(),
							gcv = s.GuestClientVersion.ToShortString(),
							hct = this.GetDurationSeconds(s.HostConnectedStartTime, utcNow),
							gct = this.GetDurationSeconds(s.GuestConnectedStartTime, utcNow),
							hcc = s.HostConnectedCount,
							gcc = s.GuestConnectedCount,
							qt = s.QueuedEventType,
							n = s.Notes,
							p = permissionSets[s.Host == userName ? 0 : 1],
							a = s.Attributes,
							clp = new
							{
								s = s.SessionID,
								i = s.Name,
								n = (permissionSets[s.Host == userName ? 0 : 1] & SessionPermissions.Join) != 0 ? ServerCryptoManager.Instance.GetAccessTokenString(s.SessionID, ProcessType.Host, userName, permissionSets[s.Host == userName ? 0 : 1], default(DateTime)) : null,
							}
						})
					};
				}
			});
		}

		public object EndGetHostSessionInfo(IAsyncResult result)
		{
			return Elsinore.ScreenConnect.Extensions.AssertNonNull(result as WaitForChangeAsyncResult).End();
		}

		public IAsyncResult BeginGetGuestSessionInfo(string sessionCode, Guid? sessionID, long version, AsyncCallback callback, object state)
		{
			return new WaitForChangeAsyncResult(callback, state, HttpContext.Current, version, delegate(long newVersion)
			{
				using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				{
					var session = default(Session);

					if (sessionID != null)
						session = sessionManager.GetSession(sessionID.Value);

					if (session == default(Session) && !string.IsNullOrEmpty(sessionCode))
						session = sessionManager.GetSession(sessionCode);

					Func<Session, object> translator = (ss => new { st = ss.SessionType, hcc = ss.HostConnectedCount, gcc = ss.GuestConnectedCount, h = ss.Host, clp = new { s = ss.SessionID, i = ss.Name } });

					return new
					{
						v = newVersion,
						hcs = sessionManager.DoCodeSessionsExist(),
						ss = (session == null ? null : translator(session)),
						lss = sessionManager.GetPublicSessions().Select(s => translator(s))
					};
				}
			});
		}

		public object EndGetGuestSessionInfo(IAsyncResult result)
		{
			return Elsinore.ScreenConnect.Extensions.AssertNonNull(result as WaitForChangeAsyncResult).End();
		}

		public void LogInitiatedJoin(Guid sessionID, string data)
		{
			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				sessionManager.AddSessionEvent(sessionID, new SessionEvent { EventType = SessionEventType.InitiatedJoin, Data = data });
		}

		public void TransferSessions(string sessionGroupName, Guid[] sessionIDs, string toHost)
		{
			this.ExecuteSessionProc(sessionGroupName, sessionIDs, PermissionInfo.TransferSessionPermission, (sm, sid, i, un) => sm.UpdateSession(un, sid, toHost));
		}

		public void AddEventToSessions(string sessionGroupName, Guid[] sessionIDs, SessionEventType eventType, string data)
		{
			var permissionName = this.GetAddEventPermissionName(eventType);
			this.ExecuteSessionProc(sessionGroupName, sessionIDs, permissionName, (sm, sid, i, un) => sm.AddSessionEvent(sid, new SessionEvent { Host = un, EventType = eventType, Data = data }));
		}

		public void RemoveNoteFromSession(string sessionGroupName, Guid sessionID, Guid eventID)
		{
			this.ExecuteSessionProc(sessionGroupName, new[] { sessionID }, PermissionInfo.RemoveNoteFromSessionPermission, (sm, sid, i, un) => sm.RemoveSessionEvent(sid, eventID));
		}

		public object GetSessionDetails(string sessionGroupName, Guid sessionID)
		{
			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
			{
				var variables = this.GetVariables(HttpContext.Current.User.Identity.Name);
				var isSessionInGroup = sessionManager.IsSessionInGroup(sessionID, sessionGroupName, variables);
				var sessionDetails = sessionManager.GetSessionDetails(sessionID);

				if (!isSessionInGroup || sessionDetails == null || sessionDetails.Session == null)
					return null;

				Permissions.AssertPermission(new SessionPermissionRequest { Name = PermissionInfo.ViewSessionGroupPermission, SessionGroupName = sessionGroupName, SessionType = sessionDetails.Session.SessionType }, true);

				return new
				{
					s = new
					{
						gna = sessionDetails.Session.GuestNetworkAddress.ToShortString(),
						gmn = sessionDetails.Session.GuestInfo.MachineName,
						gmd = sessionDetails.Session.GuestInfo.MachineDomain,
						gpn = sessionDetails.Session.GuestInfo.ProcessorName,
						gpc = sessionDetails.Session.GuestInfo.ProcessorVirtualCount,
						gtm = sessionDetails.Session.GuestInfo.SystemMemoryTotalMegabytes,
						gam = sessionDetails.Session.GuestInfo.SystemMemoryAvailableMegabytes,
						sct = sessionDetails.Session.GuestInfo.ScreenshotContentType,
						gut = this.GetDurationSeconds(sessionDetails.Session.GuestInfoUpdateTime, DateTime.UtcNow),
					},
					sc = sessionDetails.GuestScreenshotContent.SafeNav(sc => Convert.ToBase64String(sc)),
					cs = sessionDetails.Connections.Select(sc => new
					{
						pt = sc.Connection.ProcessType,
						ct = sc.Connection.ClientType,
						cv = sc.Connection.ClientVersion.ToShortString(),
						na = sc.Connection.NetworkAddress.ToString(),
						pn = sc.Connection.ParticipantName,
						es = sc.Events.Select(se => new
						{
							id = se.EventID,
							et = se.EventType,
							t = se.Time.ToLocalTime(),
							d = se.Data,
						})
					}),
					es = sessionDetails.Events.Select(se => new
					{
						id = se.EventID,
						et = se.EventType,
						t = se.Time.ToLocalTime(),
						h = se.Host,
						d = se.Data,
					})
				};
			}
		}

		public SessionGroup[] GetSessionGroups()
		{
			Permissions.AssertPermission(new PermissionRequest { Name = PermissionInfo.ManageSessionGroupsPermission }, true);

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				return sessionManager.GetSessionGroups();
		}

		public void SaveSessionGroups(SessionGroup[] sessionGroups)
		{
			Permissions.AssertPermission(new PermissionRequest { Name = PermissionInfo.ManageSessionGroupsPermission }, true);

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				sessionManager.SaveSessionGroups(sessionGroups);
		}

		public Guid CreateSession(SessionType sessionType, string name, bool isPublic, string code, string[] customPropertyValues)
		{
			var permissionName = (sessionType == SessionType.Support ? PermissionInfo.CreateSupportSessionPermission : PermissionInfo.CreateMeetingSessionPermission);
			Permissions.AssertPermission(new PermissionRequest { Name = permissionName }, true);

			var userName = HttpContext.Current.User.Identity.Name;

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				return sessionManager.CreateSession(userName, sessionType, name, userName, isPublic, code, customPropertyValues).SessionID;
		}

		public void UpdateSessions(string sessionGroupName, Guid[] sessionIDs, string[] names, bool[] isPublics, string[] codes, string[][] customPropertyValues)
		{
			this.ExecuteSessionProc(sessionGroupName, sessionIDs, PermissionInfo.EditSessionPermission, (sm, sid, i, un) => sm.UpdateSession(un, sid, names[i], isPublics[i], codes[i], customPropertyValues[i]));
		}

		public void SendEmail(string to, string subject, string body, bool isBodyHtml)
		{
			Permissions.AssertPermission(new PermissionRequest(), true);
			this.SendEmail(null, null, to, subject, body, isBodyHtml);
		}

		public void SendTestEmail(string from, string relayHost, string to)
		{
			Permissions.AssertPermission(new PermissionRequest { Name = PermissionInfo.AdministerPermission }, true);
			this.SendEmail(from, relayHost, to, global::Resources.Default.MailPanel_TestSubject, global::Resources.Default.MailPanel_TestBody, false);
		}

		void SendEmail(string overrideFrom, string overrideRelayHost, string to, string subject, string body, bool isBodyHtml)
		{
			var client = new SmtpClient();
			var mailMessage = ServerToolkit.Instance.CreateMailMessage();

			mailMessage.To.Add(to);
			mailMessage.Subject = subject;
			mailMessage.Body = body;
			mailMessage.IsBodyHtml = isBodyHtml;

			if (overrideFrom != null)
				mailMessage.From = new MailAddress(overrideFrom);

			if (overrideRelayHost != null)
				client.Host = overrideRelayHost;

			client.Send(mailMessage);
		}

		public string[] GetEligibleHosts()
		{
			Permissions.AssertPermission(new PermissionRequest(), true);

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
				return sessionManager.GetEligibleHosts();
		}

		public string GetInstallerUrl(string installerType, string name, string[] customPropertyValues)
		{
			Permissions.AssertPermission(new PermissionRequest { Name = PermissionInfo.BuildAccessInstallerPermission }, true);

			var relayUri = ServerExtensions.GetRelayUri(ConfigurationManager.AppSettings, HttpContext.Current.Request.Url, true, true);

			var clientLaunchParameters = new ClientLaunchParameters
			{
				SessionType = SessionType.Access,
				ProcessType = ProcessType.Guest,
				NameCallbackFormat = name,
				CustomPropertyValueCallbackFormats = customPropertyValues,
				Host = relayUri.Host,
				Port = relayUri.Port,
				EncryptionKey = ServerCryptoManager.Instance.PublicKey,
			};

			return "Bin/Elsinore.ScreenConnect.ClientSetup." + installerType + ClientLaunchParameters.ToQueryString(clientLaunchParameters, true);
		}

		int GetDurationSeconds(DateTime eventTime, DateTime utcNow)
		{
			if (eventTime == default(DateTime))
				return -1;

			return (int)(utcNow - eventTime).TotalSeconds;
		}

		string GetAddEventPermissionName(SessionEventType eventType)
		{
			switch (eventType)
			{
				case SessionEventType.AddedNote: return PermissionInfo.AddNoteToSessionPermission;
				case SessionEventType.QueuedReinstall: return PermissionInfo.ReinstallSessionPermission;
				case SessionEventType.QueuedUninstall: return PermissionInfo.UninstallSessionPermission;
				case SessionEventType.QueuedCommand: return PermissionInfo.RunCommandOutsideSessionPermission;
				case SessionEventType.QueuedWake: return PermissionInfo.JoinSessionPermission;
				case SessionEventType.QueuedMessage: return PermissionInfo.JoinSessionPermission;
				case SessionEventType.QueuedGuestInfoUpdate: return PermissionInfo.JoinSessionPermission;
				case SessionEventType.EndedSession: return PermissionInfo.EndSessionPermission;
				default: throw new ArgumentOutOfRangeException();
			}
		}

		Dictionary<string, string> GetVariables(string userName)
		{
			return new Dictionary<string, string> { { ServerConstants.VariableNameUserName, userName } };
		}

		void ExecuteSessionProc(string sessionGroupName, Guid[] sessionIDs, string permissionName, Proc<ISessionManagerChannel, Guid, int, string> proc)
		{
			if (sessionIDs == null)
				throw new ArgumentNullException("sessionIDs");

			using (var sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow())
			{
				var userName = HttpContext.Current.User.Identity.Name;
				var permissionEntries = Permissions.GetEntriesForUser();
				var variables = this.GetVariables(userName);
				var sessions = sessionManager.GetSessions(sessionGroupName, variables, null);

				foreach (var sessionID in sessionIDs)
				{
					var session = sessions.FirstOrDefault(ss => ss.SessionID == sessionID);

					if (session == null)
						throw new InvalidOperationException("Session not in specified group.");

					Permissions.AssertPermission(new SessionOwnershipPermissionRequest { Name = permissionName, SessionGroupName = sessionGroupName, SessionType = session.SessionType, IsOwned = session.Host == userName }, permissionEntries, true);
				}

				for (var i = 0; i < sessionIDs.Length; i++)
					proc(sessionManager, sessionIDs[i], i, userName);
			}
		}

		class WaitForChangeAsyncResult : AsyncResult<object>
		{
			public WaitForChangeAsyncResult(AsyncCallback originalCallback, object originalState, HttpContext context, long version, Func<long, object> executor)
				: base(originalCallback, originalState)
			{
				this.Context = context;
				this.Version = version;

				this.ExecuteAsync(
					delegate(AsyncCallback callback)
					{
						ServerExtensions.BeginWaitForChange(this.Version, this.ShouldContinue, 100000, callback, null);
					},
					delegate(IAsyncResult result)
					{
						if (!this.ShouldContinue())
							throw new TypeUnloadedException("Request aborted: Application restarting");

						var newVersion = ServerExtensions.EndWaitForChange(result);

						if (newVersion <= this.Version)
							return null;

						return executor(newVersion);
					}
				);
			}

			public HttpContext Context { get; private set; }
			public long Version { get; private set; }

			bool ShouldContinue()
			{
				return this.Context.Response.IsClientConnected && !ServerToolkit.Instance.IsHttpRuntimeShuttingDown();
			}
		}
	}
}