﻿using System;
using System.Collections.Specialized;
using System.Configuration;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.ServiceModel.Configuration;
using System.Net;
using System.Net.Sockets;
using System.Security.Principal;
using System.Diagnostics;
using System.Threading;
using System.Collections.Generic;
using System.Net.Security;

namespace Elsinore.ScreenConnect
{
	public partial class StatusPanel : System.Web.UI.UserControl
	{
		protected override void AddedControl(Control control, int index)
		{
			base.AddedControl(control, index);

			var stp = control as StatusTestPanel;

			if (stp != null)
			{
				stp.CssClass = "StatusTestPanel";
				stp.HeaderStyle.CssClass = "StatusTestPanelHeader";
				stp.ContentStyle.CssClass = "StatusTestPanelContent";
				stp.ExpandImageUrl = "Images/Expand.png";
				stp.CollapseImageUrl = "Images/Collapse.png";
				stp.LoadingImageUrl = "Images/ActivityIndicator.gif";

				stp.IncompleteSettings.StatusText = Resources.Default.StatusPanel_StatusIncompleteText;

				stp.ErrorSettings.HeaderStyle.CssClass = "Error";
				stp.ErrorSettings.StatusText = Resources.Default.StatusPanel_StatusErrorText;

				stp.PassedSettings.HeaderStyle.CssClass = "Passed";
				stp.PassedSettings.StatusText = Resources.Default.StatusPanel_StatusPassedText;

				stp.WarningSettings.HeaderStyle.CssClass = "Warning";
				stp.WarningSettings.StatusText = Resources.Default.StatusPanel_StatusWarningText;

				stp.FailedSettings.HeaderStyle.CssClass = "Failed";
				stp.FailedSettings.StatusText = Resources.Default.StatusPanel_StatusFailedText;
			}
		}

		protected override void OnLoad(EventArgs e)
		{
			base.OnLoad(e);

			if (this.IsPostBack)
			{
				var executors = new List<PageAsyncExecutor>();
				executors.Add(new VersionCheckExecutor { Panel = this.versionCheckPanel });
				executors.Add(new WindowsFirewallCheckExecutor { Panel = this.windowsFirewallCheckPanel });
				executors.Add(new ExternalAccessibilityCheckExecutor { Panel = this.externalAccessibilityCheckPanel });
				executors.Add(new BrowserUrlCheckExecutor { Panel = this.browserUrlCheckPanel });
				executors.Add(new WebServerExecutor { Panel = this.webServerCheckPanel });
				executors.Add(new RelayCheckExecutor { Panel = this.relayCheckPanel });
				executors.Add(new SessionManagerCheckExecutor { Panel = this.sessionManagerCheckPanel });

				foreach (var executor in executors)
				{
					if (executor.Panel.TestResult == TestResult.Incomplete)
					{
						var executorCopy = executor;

						var task = new PageAsyncTask
						(
							(o, ea, c, d) => executorCopy.Begin(c, d),
							r => executorCopy.End(r),
							r => executorCopy.Timeout(r),
							null,
							true
						);

						this.Page.RegisterAsyncTask(task);
					}
				}

				this.Page.ExecuteRegisteredAsyncTasks();
			}
		}

		protected override void OnPreRender(EventArgs e)
		{
			base.OnPreRender(e);
			this.timer.Enabled = this.Controls.OfType<StatusTestPanel>().Any(stp => stp.TestResult == TestResult.Incomplete);
			this.DataBind();
		}

		protected override bool OnBubbleEvent(object source, EventArgs args)
		{
			var commandEventArgs = args as CommandEventArgs;

			if (commandEventArgs != null && commandEventArgs.CommandName == "Retry")
			{
				var stp = ((Control)source).FindAncestor<StatusTestPanel>();
				stp.TestResult = TestResult.Incomplete;
				stp.TestData = null;
				return true;
			}
			else
			{
				return base.OnBubbleEvent(source, args);
			}
		}

		abstract class PageAsyncExecutor
		{
			public abstract IAsyncResult Begin(AsyncCallback callback, object state);
			public abstract void End(IAsyncResult result);
			public abstract void Timeout(IAsyncResult result);
			public StatusTestPanel Panel { get; set; }
		}

		abstract class PageAsyncWebRequestExecutor<T> : PageAsyncExecutor
		{
			WebRequest request;

			public abstract string Url { get; }
			public abstract NameValueCollection GetParameters();
			public abstract TestResult GetTestResult(T testData);

			public override IAsyncResult Begin(AsyncCallback callback, object state)
			{
				var parameters = this.GetParameters();
				this.request = ServerExtensions.CreateWebRequest(this.Url, parameters, 0);
				return this.request.BeginGetResponse(callback, state);
			}

			public override void End(IAsyncResult result)
			{
				try
				{
					using (var response = this.request.EndGetResponse(result))
					using (var stream = response.GetResponseStream())
						this.Panel.TestData = ServerExtensions.DeserializeXml<T>(stream);

					this.Panel.TestResult = this.GetTestResult((T)this.Panel.TestData);
				}
				catch (Exception)
				{
					this.Panel.TestResult = TestResult.Error;
				}
			}

			public override void Timeout(IAsyncResult result)
			{
				Extensions.Try(() => this.request.Abort());
				this.Panel.TestResult = TestResult.Error;
			}
		}

		class VersionCheckExecutor : PageAsyncWebRequestExecutor<VersionTestResult>
		{
			public override string Url
			{
				get { return ServerConstants.VersionCheckUrl; }
			}

			public override NameValueCollection GetParameters()
			{
				var parameters = new NameValueCollection();
				parameters.Add("Version", Constants.ProductVersion.ToString());
				parameters.Add("Platform", Environment.OSVersion.Platform.ToString());
				parameters.Add("IsPreRelease", Constants.IsPreRelease.ToString());

				var licenseManager = new LicenseManager();
				licenseManager.LoadLicensesFromDefaultPath();

				foreach (var licenseEnvelope in licenseManager.GetLicenseEnvelopes())
				{
					parameters.Add("LicenseID", licenseEnvelope.Contents.LicenseID);
					parameters.Add("LicenseVersion", licenseEnvelope.Contents.VersionString);
				}

				return parameters;
			}

			public override TestResult GetTestResult(VersionTestResult testData)
			{
				return (testData.NewVersionAvailable ? TestResult.Warning : TestResult.Passed);
			}
		}

		class WindowsFirewallCheckExecutor : PageAsyncExecutor
		{
			public override IAsyncResult Begin(AsyncCallback callback, object state)
			{
				var exeName = Process.GetCurrentProcess().MainModule.FileName;
				var webServerPort = ConfigurationManager.AppSettings.GetUriPort(ServerConstants.WebServerListenUriSettingsKey);
				var relayPort = ConfigurationManager.AppSettings.GetUriPort(ServerConstants.RelayListenUriSettingsKey);

				var result = new AsyncResult<object> { OriginalCallback = callback, OriginalState = state };

				try
				{
					var testResult = new WindowsFirewallTestResult();
					testResult.IsFirewallEnabled = ServerToolkit.Instance.IsWindowsFirewallEnabled();
					testResult.IsWebServerPortAllowed = ServerToolkit.Instance.IsWindowsFirewallTcpPortAllowed(exeName, webServerPort);
					testResult.IsRelayPortAllowed = ServerToolkit.Instance.IsWindowsFirewallTcpPortAllowed(exeName, relayPort);
					result.ReturnValue = testResult;
				}
				catch (Exception ex)
				{
					result.Exception = ex;
				}

				result.Complete(true);
				return result;
			}

			public override void End(IAsyncResult result)
			{
				var testResult = ((AsyncResult<object>)result).ReturnValue as WindowsFirewallTestResult;

				if (testResult != null)
				{
					this.Panel.TestResult = testResult.IsWebServerPortAllowed && testResult.IsRelayPortAllowed ? TestResult.Passed : TestResult.Failed;
					this.Panel.TestData = testResult;
				}
				else
				{
					this.Panel.TestResult = TestResult.Error;
				}
			}

			public override void Timeout(IAsyncResult result)
			{
				// will never occur
			}
		}

		class ExternalAccessibilityCheckExecutor : PageAsyncWebRequestExecutor<ExternalAccessibilityTestResult>
		{
			public override string Url
			{
				get { return ServerConstants.AccessibilityCheckUrl; }
			}

			public override NameValueCollection GetParameters()
			{
				return ServerExtensions.GetExternalAccessibilityCheckParameters(ConfigurationManager.AppSettings, this.Panel.Page.Request.Url);
			}

			public override TestResult GetTestResult(ExternalAccessibilityTestResult testData)
			{
				return (testData.WebServerErrorMessage == null && testData.RelayErrorMessage == null ? TestResult.Passed : TestResult.Failed);
			}
		}

		class BrowserUrlCheckExecutor : PageAsyncWebRequestExecutor<BrowserUrlTestResult>
		{
			public override string Url
			{
				get { return ServerConstants.BrowserUrlCheckUrl; }
			}

			public override NameValueCollection GetParameters()
			{
				var parameters = new NameValueCollection();
				parameters["Host"] = this.Panel.Page.Request.Url.Host;
				return parameters;
			}

			public override TestResult GetTestResult(BrowserUrlTestResult testData)
			{
				return (testData.IsHostResolvable ? TestResult.Passed : TestResult.Warning);
			}
		}

		class WebServerExecutor : PageAsyncExecutor
		{
			WebRequest request;

			public override IAsyncResult Begin(AsyncCallback callback, object state)
			{
				this.Panel.TestData = ServerExtensions.GetWebServerUri(ConfigurationManager.AppSettings, null, false, false).Uri.AbsoluteUri;
				this.request = ServerExtensions.CreateWebRequest((string)this.Panel.TestData, null, 0);
				ServicePointManager.ServerCertificateValidationCallback = (RemoteCertificateValidationCallback)delegate { return true; };
				return request.BeginGetResponse(callback, state);
			}

			public override void End(IAsyncResult result)
			{
				try
				{
					using (this.request.EndGetResponse(result))
						this.Panel.TestResult = TestResult.Passed;
				}
				catch (WebException)
				{
					this.Panel.TestResult = TestResult.Failed;
				}
				catch (Exception)
				{
					this.Panel.TestResult = TestResult.Error;
				}
				finally
				{
					ServicePointManager.ServerCertificateValidationCallback = null;
				}
			}

			public override void Timeout(IAsyncResult result)
			{
				Extensions.Try(() => this.request.Abort());
				this.Panel.TestResult = TestResult.Error;
			}
		}

		class RelayCheckExecutor : PageAsyncExecutor
		{
			Socket socket;

			public override IAsyncResult Begin(AsyncCallback callback, object state)
			{
				var uriBuilder = ConfigurationManager.AppSettings.GetUri(ServerConstants.RelayListenUriSettingsKey);

				if (string.IsNullOrEmpty(uriBuilder.Host) || IPAddress.Parse(uriBuilder.Host).IsAny())
					uriBuilder.Host = IPAddress.Loopback.ToString();

				this.Panel.TestData = uriBuilder.Uri;

				var address = IPAddress.Parse(uriBuilder.Host);
				this.socket = new Socket(address.AddressFamily, SocketType.Stream, ProtocolType.Tcp);
				return this.socket.BeginConnect(address, uriBuilder.Port, callback, state);
			}

			public override void End(IAsyncResult result)
			{
				try
				{
					this.socket.EndConnect(result);
					this.Panel.TestResult = TestResult.Passed;
				}
				catch (SocketException)
				{
					this.Panel.TestResult = TestResult.Failed;
				}
				catch (Exception)
				{
					this.Panel.TestResult = TestResult.Error;
				}
				finally
				{
					this.socket.ShutdownAndDisposeQuietly();
				}
			}

			public override void Timeout(IAsyncResult result)
			{
				this.socket.ShutdownAndDisposeQuietly();
				this.Panel.TestResult = TestResult.Error;
			}
		}

		class SessionManagerCheckExecutor : PageAsyncExecutor
		{
			ISessionManagerChannel sessionManager;

			public override IAsyncResult Begin(AsyncCallback callback, object state)
			{
				this.sessionManager = ServiceChannelPool<ISessionManagerChannel>.Instance.Borrow();
				return sessionManager.BeginWaitForChange(0, 0, 0, callback, state);
			}

			public override void End(IAsyncResult result)
			{
				try
				{
					this.Panel.TestData = WebConfigurationManager.GetSection<ClientSection>().Endpoints.OfType<ChannelEndpointElement>().Where(ep => ep.Contract == typeof(ISessionManagerChannel).FullName).Select(ep => ep.Address.ToString()).FirstOrDefault();
					this.sessionManager.EndWaitForChange(result);
					this.Panel.TestResult = TestResult.Passed;
				}
				catch (Exception)
				{
					this.Panel.TestResult = TestResult.Failed;
				}
				finally
				{
					this.sessionManager.DisposeQuietly();
				}
			}

			public override void Timeout(IAsyncResult result)
			{
				this.sessionManager.DisposeQuietly();
				this.Panel.TestResult = TestResult.Error;
			}
		}
	}
}