#!/bin/bash

if [ $UID -ne 0 ]; then
	# Clear out any existing credentials so the user always gets a prompt
	sudo -K
	echo "Type your password below to give ScreenConnect permission to install: "
	echo "(You can also Ctrl-C to exit and re-run the script as root)"
	sudo "$0"
	exit 0
fi

INSTALL_SCRIPT_PATH="$(dirname "$0")"
INSTALL_DIRECTORY="$INSTALL_SCRIPT_PATH/Installer"
SCREENCONNECT_DIRECTORY="$INSTALL_SCRIPT_PATH/ScreenConnect"

source "$INSTALL_DIRECTORY/functions.sh"

if [ "$BASH_VERSION" = "" ]; then
	echo Required bash scripting environment not detected. You are likely running with /bin/sh. Rerun with /bin/bash?
	if prompt_yesno; then /bin/bash "$0"; fi
	exit 0
fi

echo ""
echo "Welcome to the ScreenConnect Installer"
echo ""
echo "The installer will do these things:"
echo "1) Prompt you for installation options"
echo "2) Display a list of actions to be taken"
echo "3) Prompt you for execution of the actions"
echo "4) Execute the actions"

find "$INSTALL_DIRECTORY/Distributions" -name "*.sh" | sort > /tmp/distros

while read distroFile; do
	source "$distroFile"
		
	if is_distro_valid; then
		break;
	fi
done < /tmp/distros

rm /tmp/distros
	
# This is filled by the base dependencies function (or whatever a script adds in dependency_check()
# It consists of dependencies that need to be addressed in the form dependency:reason
problematic_dependencies=()
# These actions are used by the build_install_actions script
installer_actions=()
installer_action_descriptions=()

# Execute the actual dependency check, this will fill problematic_dependencies
dependency_check
	
# Loop through the missing dependencies and build the installer action and description arrays
for i in ${!problematic_dependencies[*]}
do
	# Each element is dependency:reason
	dependency_parts=($(echo "${problematic_dependencies[$i]}"))
	build_dependency_action "${dependency_parts[0]}"
		
	if [ $? == 1 ]; then
		# Return value of 1 means a build action was added, so remove the element from the array
		unset problematic_dependencies[$i]
	fi
done

if [ ${#problematic_dependencies[*]} -gt 0 ]; then
	echo ""
	echo "We were unable to resolve these dependencies:"
		
	is_a_dependency_critical=0

	for i in ${!problematic_dependencies[*]}
	do
		echo ${problematic_dependencies[$i]}
		(echo ${problematic_dependencies[$i]} | grep "(critical)" > /dev/null) && is_a_dependency_critical=1
	done
		
	if [ $is_a_dependency_critical -eq 1 ]; then
		echo ""
		echo "One or more critical dependencies could not be resolved. Installation cannot continue."
		exit 1
	fi

	echo ""
	echo "None of these are required. Would you like to continue?"

	if ! prompt_yesno; then
		echo "Exiting due to unresolved dependencies..."
		exit 1
	fi
fi

# Prompt user for options
installation_directory="/opt/ScreenConnect"

while true; do
	echo ""
	echo "Where would you like to install ScreenConnect?"
	echo -n "[$installation_directory] "
	read input

	if [ "$input" == "" ]; then
		break;
	elif [[ "$input" == "/"* ]]; then
		installation_directory=$input
		break;
	else
		echo ""
		echo "Invalid location"
	fi
done

is_upgrade=0

if test -d $installation_directory; then is_upgrade=1; fi

if [ $is_upgrade -eq 1 ]; then
	# HACK to determine old version ... really should be able to inspect the binaries
	# but there are no built-in tools that can do this reliably
	old_version=4.0
	(cat "$installation_directory/App_GlobalResources/Default.resx" 2> /dev/null | grep "ChangePasswordPanel" > /dev/null 2>&1) && old_version=3.2
	(cat "$installation_directory/App_GlobalResources/Default.resx" 2> /dev/null | grep "Client.ApplicationTitle" > /dev/null 2>&1) && old_version=3.0
	(cat "$installation_directory/App_GlobalResources/Default.resx" 2> /dev/null | grep "CreateSessionPanel.AutoJoinMessage" > /dev/null 2>&1) && old_version=2.5

	echo ""
	echo "It appears that there is already an installation at $installation_directory ($old_version or higher)"
	echo "Would you like to upgrade?"

	if ! prompt_yesno; then
		echo "Installation directory already exists, and you chose not to upgrade it. Please erase the existing installation to perform a fresh install in $installation_directory or rerun this script and choose to upgrade"
		exit 1
	fi
fi
	
# Setup environment because we have to call mono
export PATH=$installation_directory/App_Runtime/bin:$PATH
export MONO_PATH=$installation_directory/App_Runtime/lib
export MONO_CFG_DIR=$installation_directory/App_Runtime/etc

# Find init.d or whatever is appropriate for this distro
initialize_startup_directory_information

service_name=""
	
# Stop any running services and remove any existing service startup scripts
# (That point to the install folder) if upgrading
# This also gets the service name
if [ $is_upgrade -eq 1 ]; then
	service_names=()
	get_existing_service_names
		
	if [ ${#service_names[*]} -gt 0 ]; then
		for cur_service_name in ${service_names[*]}
		do
			build_stop_and_remove_service_action "$cur_service_name"
		done
			
		service_name=${service_names[0]}
	fi
fi

if [ "$service_name" == "" ]; then
	service_name=$(basename "$installation_directory" | tr "[A-Z]" "[a-z]")
fi

# Determining name of init.d script / service name
echo ""
echo "What would you like as the service name for this ScreenConnect installation?"
echo -n "[$service_name] "
read input

if [ "$input" != "" ]; then
	service_name=$input
fi

# Build up a list of what we need to do
# Back up files if upgrading
if [ $is_upgrade -eq 1 ]; then
	installer_actions[${#installer_actions[*]}]="make_backup"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Back up the existing configuration"
fi

# Run distribution-specific function to add the build actions to do startup scripts
build_startup_actions

mono_executable_file_path=$(get_mono_executable_file_path "$INSTALL_DIRECTORY")

# Copy installation files
installer_actions[${#installer_actions[*]}]="
	mkdir -p \"$installation_directory\" &>/dev/null;
	cp -r \"$SCREENCONNECT_DIRECTORY/\"* \"$installation_directory/\";
	cp \"$mono_executable_file_path\" \"$installation_directory/App_Runtime/bin/mono\";
	chmod 755 \"$installation_directory/App_Runtime/bin/\"*;
"
installer_action_descriptions[${#installer_action_descriptions[*]}]="Copy files into $installation_directory"
	
# Determine configuration for xsl transform
configuration="Release"
if ls "$SCREENCONNECT_DIRECTORY"/Bin/*.mdb &>/dev/null; then
	configuration="Debug"
fi

# Restore backed up files if upgrading
if [ $is_upgrade -eq 1 ]; then
	installer_actions[${#installer_actions[*]}]="restore_backup"
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Restore backed up configuration"
fi

# Do the XSL transforms
installer_actions[${#installer_actions[*]}]="
	cd \"$INSTALL_DIRECTORY\";
	mono Xsl.exe \"$installation_directory/App_ClientConfig/Elsinore.ScreenConnect.Client.resx\" Elsinore.ScreenConnect.Client.resx.xsl \"$installation_directory/App_ClientConfig/Elsinore.ScreenConnect.Client.resx\" oldVersion=$old_version \"defaultResxPath=$installation_directory/App_GlobalResources/Default.resx\";
	mono Xsl.exe \"$installation_directory/web.config\" web.config.xsl \"$installation_directory/web.config\" configuration=$configuration platform=Mono;
	mono Xsl.exe \"$installation_directory/App_GlobalResources/Default.resx\" Default.resx.xsl \"$installation_directory/App_GlobalResources/Default.resx\" oldVersion=$old_version;
	mono Xsl.exe \"$installation_directory/App_Data/SessionGroup.xml\" SessionGroup.xml.xsl \"$installation_directory/App_Data/SessionGroup.xml\";
	if [ -f \"$installation_directory/App_Data/Role.xml\" ]; then mono Xsl.exe \"$installation_directory/App_Data/Role.xml\" Role.xml.xsl \"$installation_directory/App_Data/Role.xml\" oldVersion=$old_version; fi;
"
if [ $is_upgrade -ne 1 ]; then
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Initialize configuration files"
else
	installer_action_descriptions[${#installer_action_descriptions[*]}]="Upgrade configuration files"
fi

# Start screenconnect
build_start_service_action

# Confirm the what we're going to do
echo ""
echo "The installation will perform the following actions:"

for i in ${!installer_action_descriptions[*]}
do
	echo "- ${installer_action_descriptions[i]}"
done

echo ""
echo "Do you want to install ScreenConnect?"

if prompt_yesno; then
	echo ""

	# Do the install
	for i in ${!installer_actions[*]}
	do
		echo "Running '${installer_action_descriptions[i]}'..."
		eval ${installer_actions[i]}
	done

	echo ""
	echo "Installation complete!"
	echo ""
	echo "Trying to figure out the best URL for you to use..."
	echo ""
	
	launch_web_page "$(mono "$installation_directory/Bin/Elsinore.ScreenConnect.Service.exe" launch host)"

	echo ""
	exit 0
else
	echo "You chose not to install, your system has not been modified."
fi
